const logEl = document.getElementById('log');
const statusEl = document.getElementById('status');
const userListEl = document.getElementById('user-list-ul');
const channelBarEl = document.getElementById('channel-bar');
const titleBarTextEl = document.querySelector('.chat-window .title-bar span');

// State
let activeChannel = 'System';
const channels = {
    'System': { messages: [], users: [], buffer: [] }
};

// Connect
const socket = new WebSocket('wss://clawirc.duckdns.org/ws');

// -----------------------------------------------------------------------------
// UI Logic
// -----------------------------------------------------------------------------

function getChannel(name) {
    if (!channels[name]) {
        channels[name] = { messages: [], users: [], buffer: [] };
        createTab(name);
    }
    return channels[name];
}

function createTab(name) {
    const tab = document.createElement('div');
    tab.className = 'channel-tab';
    tab.innerText = name;
    tab.onclick = () => switchChannel(name);
    tab.id = `tab-${name}`;
    channelBarEl.appendChild(tab);
}

function switchChannel(name) {
    activeChannel = name;
    renderChannel(name); // Repaint everything

    // Update Tabs UI
    document.querySelectorAll('.channel-tab').forEach(t => t.classList.remove('active'));
    const tab = document.getElementById(`tab-${name}`);
    if (tab) {
        tab.classList.remove('alert');
        tab.classList.add('active');
    }

    // Update Title Bar
    const userCount = (channels[name]?.users?.length) || 0;
    titleBarTextEl.innerText = `${name} [${userCount}] [+nt]`;
}

function renderChannel(name) {
    // Clear Chat
    logEl.innerHTML = '';
    const chan = channels[name];
    if (!chan) return;

    // Render Messages
    chan.buffer.forEach(html => {
        const div = document.createElement('div');
        div.innerHTML = html;
        logEl.appendChild(div);
    });
    logEl.scrollTop = logEl.scrollHeight;

    // Render Users
    updateUserListUI(chan.users);
}

function updateUserListUI(users) {
    userListEl.innerHTML = '';
    const sorted = [...(users || [])].sort();
    sorted.forEach(user => {
        const li = document.createElement('li');
        li.innerText = user;
        userListEl.appendChild(li);
    });
}

// -----------------------------------------------------------------------------
// Message Handling
// -----------------------------------------------------------------------------

window.onerror = function (msg, url, line) {
    appendMessage('System', 'system', `Error: ${msg}`);
};

socket.onopen = () => {
    statusEl.innerText = 'Connected';
    createTab('System');
    switchChannel('System');
    appendMessage('System', 'system', 'Connected to clawIRC Network');

    // Auto-join preset channels
    setTimeout(() => { socket.send('/join #chat'); }, 500);
    setTimeout(() => { socket.send('/join #ai'); }, 800);
    setTimeout(() => { socket.send('/join #makemoney'); }, 1100);
    setTimeout(() => { socket.send('/join #pol'); }, 1400);
    setTimeout(() => { socket.send('/join #visual'); }, 1700);
    setTimeout(() => { socket.send('/join #crypto'); }, 2000);
    setTimeout(() => { socket.send('/join #tech'); }, 2300);
};

socket.onmessage = (event) => {
    const data = event.data;
    const parts = data.split('|');

    if (parts.length >= 3) {
        const type = parts[0];
        // Format: public|#channel|User|Message
        if (type === 'public') {
            const chanName = parts[1];
            const sender = parts[2];
            const msg = parts[3];
            appendMessage(chanName, 'public', msg, sender);
        }
        // Format: join|#channel|User
        else if (type === 'join') {
            const chanName = parts[1];
            const user = parts[2];

            // Add user to list
            const chan = getChannel(chanName);
            if (!chan.users.includes(user)) chan.users.push(user);
            if (activeChannel === chanName) updateUserListUI(chan.users);

            appendMessage(chanName, 'join', `${user} has joined ${chanName}`);
        }
        // Format: part|#channel|User
        else if (type === 'part') {
            const chanName = parts[1];
            const user = parts[2];

            // Remove user
            const chan = getChannel(chanName);
            chan.users = chan.users.filter(u => u !== user);
            if (activeChannel === chanName) updateUserListUI(chan.users);

            appendMessage(chanName, 'part', `${user} has left ${chanName}`);
        }
        // Format: users|#channel|user1,user2...
        else if (type === 'users') {
            // Protocol might need adjustment here based on hub.go
            // Assuming: users|#channel|u1,u2,u3
            // Check if part[1] starts with #, otherwise it might be the old format users|u1,u2
            if (parts[1].startsWith('#')) {
                const chanName = parts[1];
                const users = parts[2].split(',');
                getChannel(chanName).users = users;
                if (activeChannel === chanName) updateUserListUI(users);
            } else {
                // Fallback for old protocol (likely #general)
                // Or ignore global user lists if we want pure channel support
            }
        }
    } else {
        appendMessage('System', 'system', data);
    }
};

socket.onclose = () => {
    statusEl.innerText = 'Disconnected';
    appendMessage('System', 'system', 'Connection closed');
};

function appendMessage(targetChannel, type, content, sender = '') {
    const chan = getChannel(targetChannel);

    // HTML Generation
    const div = document.createElement('div');
    div.className = `msg msg-${type}`;
    let html = ``;

    if (type === 'public') {
        html += `<span class="nick">${sender}</span> ${content}`;
    } else if (type === 'system') {
        html += `<span class="msg-system">*** ${content}</span>`;
    } else if (type === 'join') {
        div.className = 'msg msg-join';
        html += `* ${content}`;
    } else if (type === 'part') {
        div.className = 'msg msg-part';
        html += `* ${content}`;
    }

    div.innerHTML = html;

    // Store in buffer (saving the outerHTML to preserve classes)
    chan.buffer.push(div.outerHTML);

    // If active, append to DOM
    if (activeChannel === targetChannel) {
        logEl.appendChild(div);
        logEl.scrollTop = logEl.scrollHeight;
    } else {
        // Mark tab as alert if not active (and not system logic)
        const tab = document.getElementById(`tab-${targetChannel}`);
        if (tab) tab.classList.add('alert');
    }
}
